/*
 * Decompiled with CFR 0.152.
 */
package cz.snyll.sunny.services;

import cz.snyll.sunny.domain.EventEntry;
import cz.snyll.sunny.domain.InfoData;
import cz.snyll.sunny.domain.inverter.Inverter;
import cz.snyll.sunny.repositories.InfoDataRepository;
import cz.snyll.sunny.services.EventEntryManagerService;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import jakarta.transaction.Transactional;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Date;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;

@Service
public class InfoDataManagerService {
    private final InfoDataRepository infoDataRepository;
    private final EventEntryManagerService eventEntryManagerService;
    private final Map<String, InfoData> inMemoryStore = new ConcurrentHashMap();

    public InfoDataManagerService(InfoDataRepository infoDataRepository, EventEntryManagerService eventEntryManagerService) {
        this.infoDataRepository = infoDataRepository;
        this.eventEntryManagerService = eventEntryManagerService;
    }

    @PostConstruct
    public void initCacheFromDb() {
        Iterable allFromDb = this.infoDataRepository.findAll();
        for (InfoData info : allFromDb) {
            this.inMemoryStore.put(info.getDataKey(), info);
        }
    }

    public InfoData findByDataKey(String dataKey) {
        InfoData inMemoryData = (InfoData)this.inMemoryStore.get(dataKey);
        if (inMemoryData != null) {
            return inMemoryData;
        }
        InfoData dbData = this.infoDataRepository.findByDataKey(dataKey);
        if (dbData != null) {
            this.inMemoryStore.putIfAbsent(dataKey, dbData);
            return dbData;
        }
        return null;
    }

    public Iterable<InfoData> findAll() {
        return new ArrayList<InfoData>(this.inMemoryStore.values());
    }

    public InfoData save(InfoData infoData, boolean flushToDb) {
        if (flushToDb) {
            this.infoDataRepository.save((Object)infoData);
        }
        this.inMemoryStore.put(infoData.getDataKey(), infoData);
        return infoData;
    }

    public void deleteByDataKey(String dataKey) {
        this.inMemoryStore.remove(dataKey);
    }

    @Transactional
    @Scheduled(fixedDelay=3600000L)
    public void flushToDatabaseAndClean() {
        LocalDateTime minus48hours = LocalDateTime.now().minusHours(48L);
        Date dateMinus48Hours = Date.from(minus48hours.atZone(ZoneId.systemDefault()).toInstant());
        this.inMemoryStore.values().removeIf(info -> info.getDataFreshness().before(dateMinus48Hours));
        int successCount = 0;
        int errorCount = 0;
        for (InfoData infoData : this.inMemoryStore.values()) {
            try {
                InfoData existing = this.infoDataRepository.findByDataKey(infoData.getDataKey());
                if (existing != null) {
                    existing.setDataValue(infoData.getDataValue());
                    existing.setUnits(infoData.getUnits());
                    existing.setDataFreshness(infoData.getDataFreshness());
                    existing.setValidTime(infoData.getValidTime());
                    this.infoDataRepository.save((Object)existing);
                    this.inMemoryStore.put(existing.getDataKey(), existing);
                } else {
                    infoData.setId(null);
                    InfoData saved = (InfoData)this.infoDataRepository.save((Object)infoData);
                    this.inMemoryStore.put(saved.getDataKey(), saved);
                }
                ++successCount;
            }
            catch (Exception e) {
                if (e.getMessage().contains("UNIQUE constraint failed")) {
                    try {
                        InfoData existing = this.infoDataRepository.findByDataKey(infoData.getDataKey());
                        if (existing != null) {
                            existing.setDataValue(infoData.getDataValue());
                            existing.setUnits(infoData.getUnits());
                            existing.setDataFreshness(infoData.getDataFreshness());
                            existing.setValidTime(infoData.getValidTime());
                            this.infoDataRepository.save((Object)existing);
                            this.inMemoryStore.put(existing.getDataKey(), existing);
                            ++successCount;
                            continue;
                        }
                        ++errorCount;
                    }
                    catch (Exception retryException) {
                        ++errorCount;
                    }
                    continue;
                }
                ++errorCount;
            }
        }
        String message = String.format("Performed InfoData flush to DB + cleanup (older than 48h). Success: %d, Errors: %d", successCount, errorCount);
        this.eventEntryManagerService.raiseEvent(message, errorCount > 0 ? EventEntry.EventType.WARNING : EventEntry.EventType.SUCCESS, 1440);
    }

    @PreDestroy
    public void onShutdown() {
        this.flushToDatabaseAndClean();
    }

    public Set<InfoData> findByPrefix(String prefix) {
        return this.inMemoryStore.values().stream().filter(info -> info.getDataKey().startsWith(prefix)).collect(Collectors.toSet());
    }

    public InfoData getInfoDataForInverter(Inverter inverter, String key) {
        String inverterKey = "inverter_" + inverter.getId() + "_" + key;
        InfoData retrievedInfoData = this.findByDataKey(inverterKey);
        if (retrievedInfoData == null || !retrievedInfoData.isValid()) {
            retrievedInfoData = this.findByDataKey(key);
        }
        return retrievedInfoData;
    }
}

