/*
 * Decompiled with CFR 0.152.
 */
package cz.snyll.sunny.services;

import cz.snyll.sunny.domain.Device;
import cz.snyll.sunny.domain.DeviceSetting;
import cz.snyll.sunny.domain.DeviceStatus;
import cz.snyll.sunny.domain.EventEntry;
import cz.snyll.sunny.repositories.DeviceRepository;
import cz.snyll.sunny.repositories.DeviceStatusRepository;
import cz.snyll.sunny.services.DeviceControllers.DeviceController;
import cz.snyll.sunny.services.DeviceControllers.DeviceControllerFactory;
import cz.snyll.sunny.services.DeviceControllers.DimmerController;
import cz.snyll.sunny.services.DeviceControllers.ShellyRestService;
import cz.snyll.sunny.services.DeviceManager;
import cz.snyll.sunny.services.DeviceSettingManagerService;
import cz.snyll.sunny.services.EventEntryManagerService;
import jakarta.transaction.Transactional;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.temporal.ChronoField;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class DeviceManagerService
implements DeviceManager {
    private static final Logger log = LoggerFactory.getLogger(DeviceManagerService.class);
    private final ShellyRestService shellyRestService;
    private DeviceSettingManagerService deviceSettingManagerService;
    private EventEntryManagerService eventEntryManagerService;
    private DeviceRepository deviceRepository;
    private DeviceStatusRepository deviceStatusRepository;

    @Autowired
    public DeviceManagerService(DeviceRepository deviceRepository, EventEntryManagerService eventEntryManagerService, DeviceSettingManagerService deviceSettingManagerService, DeviceStatusRepository deviceStatusRepository, ShellyRestService shellyRestService) {
        this.deviceSettingManagerService = deviceSettingManagerService;
        this.eventEntryManagerService = eventEntryManagerService;
        this.deviceRepository = deviceRepository;
        this.deviceStatusRepository = deviceStatusRepository;
        this.shellyRestService = shellyRestService;
    }

    public void saveDevice(Device device) {
        if (device.getDeviceStatus() == null) {
            DeviceStatus deviceStatus = new DeviceStatus();
            deviceStatus.setLastDayTotalConsumption(0.0f);
            deviceStatus.setTodayConsumption(0.0f);
            deviceStatus.setTotalConsumption(0.0f);
            deviceStatus.setCurrentState(DeviceStatus.OperationStatus.OFF);
            deviceStatus.setLastStateChange(LocalDateTime.now());
            deviceStatus.setDailyResetHappened(LocalDate.now().minusDays(1L));
            deviceStatus.setDevice(device);
            device.setDeviceStatus(deviceStatus);
        }
        if (device.getId() == null) {
            this.eventEntryManagerService.raiseEvent("DEVICE: Adding new device with name: " + device.getDeviceName(), EventEntry.EventType.SUCCESS);
        }
        this.eventEntryManagerService.raiseEvent("DEVICE: Adding new device with name: " + device.getDeviceName() + "; id " + device.getId(), EventEntry.EventType.SUCCESS);
        Device savedDevice = (Device)this.deviceRepository.save((Object)device);
        if (savedDevice.getId() != null && savedDevice.getDeviceStatus().getLastDayTotalConsumption() == 0.0f) {
            try {
                DeviceController deviceController = DeviceControllerFactory.createDeviceController((Device)savedDevice, (EventEntryManagerService)this.eventEntryManagerService, (ShellyRestService)this.shellyRestService);
                if (deviceController != null) {
                    deviceController.refreshData();
                    float hardwareTotal = deviceController.getTotalConsumption();
                    if (hardwareTotal > 0.0f) {
                        savedDevice.getDeviceStatus().setLastDayTotalConsumption(hardwareTotal);
                        this.deviceStatusRepository.save((Object)savedDevice.getDeviceStatus());
                        log.info("Initialized lastDayTotalConsumption for device {} with hardware counter value: {} kWh", (Object)savedDevice.getDeviceName(), (Object)Float.valueOf(hardwareTotal));
                    }
                }
            }
            catch (Exception e) {
                log.warn("Could not initialize hardware counter baseline for device {}: {}", (Object)savedDevice.getDeviceName(), (Object)e.getMessage());
            }
        }
    }

    public void saveDeviceStatus(DeviceStatus deviceStatus) {
        DeviceStatus savedDeviceStatus = (DeviceStatus)this.deviceStatusRepository.save((Object)deviceStatus);
    }

    public void deleteDevice(Device device) {
        this.deviceStatusRepository.deleteById((Object)device.getDeviceStatus().getId());
        this.deviceRepository.deleteById(device.getId());
        this.eventEntryManagerService.raiseEvent("DEVICE: Device deleted. Name: " + device.getDeviceName(), EventEntry.EventType.WARNING);
    }

    public void deleteDeviceById(long id) {
        this.deviceRepository.deleteById(Long.valueOf(id));
        this.eventEntryManagerService.raiseEvent("DEVICE: Device deleted. ID: " + id, EventEntry.EventType.WARNING);
    }

    public Device loadDevice(Long id) {
        return this.deviceRepository.findById((Object)id).orElse(null);
    }

    public List<Device> loadAllDimmableDevices() {
        ArrayList allDevices = new ArrayList();
        this.deviceRepository.findAll().forEach(allDevices::add);
        return allDevices.stream().filter(device -> {
            DeviceController controller = DeviceControllerFactory.createDeviceController((Device)device, (EventEntryManagerService)this.eventEntryManagerService, (ShellyRestService)this.shellyRestService);
            return controller instanceof DimmerController;
        }).collect(Collectors.toList());
    }

    public ArrayList<Device> loadAll() {
        return (ArrayList)this.deviceRepository.findAll();
    }

    public boolean shouldDeviceTurnOn(Device device) {
        Duration runningTime = Duration.ZERO;
        if (device.getDeviceStatus().getCurrentState() == DeviceStatus.OperationStatus.ON) {
            log.info("Device is already on, checking if it should be turned off");
            runningTime = Duration.between(device.getDeviceStatus().getLastStateChange(), LocalDateTime.now());
        }
        log.info("Checking if device should be turned on: {}, today consumption: {}, today runtime: {}, maxruntime: {}, minruntime: {}, runningTime: {}, currentruntime: {}", new Object[]{device.getDeviceName(), Float.valueOf(device.getDeviceStatus().getTodayConsumption()), device.getDeviceStatus().getTodayRuntime(), device.getMaxRuntime(), device.getMinRuntime(), runningTime.toMinutes(), device.getDeviceStatus().getCurrentRuntime()});
        if (device.getDeviceStatus() == null) {
            this.eventEntryManagerService.raiseEvent("Inconsistent state of device " + device.getDeviceName() + ". It seems not to have any Device Status assigned.", EventEntry.EventType.ERROR, 10);
            return false;
        }
        if (device.getDeviceSettings() == null) {
            log.info("Device has no settings, skipping");
            return false;
        }
        if (device.getDeviceStatus().getTodayConsumption() >= device.getMaxDailyPowerConsumption() && device.getMaxDailyPowerConsumption() > 0.0f) {
            log.info("Device has reached max daily power consumption, skipping");
            return false;
        }
        if (device.getDeviceStatus().getTodayRuntime() + device.getDeviceStatus().getCurrentRuntime() >= (long)device.getMaxRuntimeDaily() && device.getMaxRuntimeDaily() > 0) {
            log.info("Device has reached max daily runtime, skipping");
            return false;
        }
        if (runningTime.toMinutes() >= (long)device.getMaxRuntime() && device.getMaxRuntime() > 0) {
            this.eventEntryManagerService.raiseEvent("DEVICE: Device " + device.getDeviceName() + " has reached max runtime, skipping", EventEntry.EventType.INFO, 10);
            log.info("Device has reached max runtime, skipping");
            return false;
        }
        if (runningTime.toMinutes() < (long)device.getMinRuntime() && device.getMinRuntime() > 0 && device.getDeviceStatus().getCurrentState() == DeviceStatus.OperationStatus.ON) {
            this.eventEntryManagerService.raiseEvent("DEVICE: Device " + device.getDeviceName() + " has not reached min runtime, skipping", EventEntry.EventType.INFO, 10);
            log.info("Device has not reached min runtime, skipping");
            return true;
        }
        for (DeviceSetting deviceSetting : device.getDeviceSettings()) {
            log.info("Evaluating device setting: {}", (Object)deviceSetting);
            if (!this.deviceSettingManagerService.evaluateDeviceSetting(deviceSetting)) continue;
            log.info("Device setting is true, turning on device");
            return true;
        }
        return false;
    }

    public boolean manualToggle(Device device) {
        if (device.getDeviceStatus().isManualOverride()) {
            this.resetOverride(device);
            return true;
        }
        if (device.getDeviceStatus().getCurrentState() == DeviceStatus.OperationStatus.ON) {
            return this.turnOffDevice(device, true);
        }
        if (device.getDeviceStatus().getCurrentState() == DeviceStatus.OperationStatus.OFF) {
            return this.turnOnDevice(device, true);
        }
        return false;
    }

    public void resetOverride(Device device) {
        device.getDeviceStatus().setOverriddenUntil(LocalDateTime.now().minusHours(1L));
        device.getDeviceStatus().setManualOverride(false);
        this.saveDeviceStatus(device.getDeviceStatus());
    }

    public boolean turnOnDevice(Device device, boolean manual) {
        try {
            DeviceController deviceController = DeviceControllerFactory.createDeviceController((Device)device, (EventEntryManagerService)this.eventEntryManagerService, (ShellyRestService)this.shellyRestService);
            if (deviceController == null) {
                this.eventEntryManagerService.raiseEvent("DEVICE: Error while getting Device controller: " + device.getDeviceName() + " - no controller found", EventEntry.EventType.ERROR, 60);
                return false;
            }
            if (deviceController.turnOnDevice()) {
                if (device.getDeviceStatus().getCurrentState() == DeviceStatus.OperationStatus.OFF) {
                    device.getDeviceStatus().setLastStateChange(LocalDateTime.now());
                    device.getDeviceStatus().setCurrentRuntime(0L);
                }
                device.getDeviceStatus().setCurrentState(DeviceStatus.OperationStatus.ON);
                if (manual) {
                    device.getDeviceStatus().setManualOverride(true);
                    device.getDeviceStatus().setOverriddenUntil(LocalDateTime.now().plusHours(1L));
                }
                this.saveDeviceStatus(device.getDeviceStatus());
                return true;
            }
            device.getDeviceStatus().setCurrentState(DeviceStatus.OperationStatus.ERROR);
            this.saveDeviceStatus(device.getDeviceStatus());
            return false;
        }
        catch (Exception e) {
            this.eventEntryManagerService.raiseEvent("DEVICE: Error while turning on device: " + device.getDeviceName() + " " + e.getMessage(), EventEntry.EventType.ERROR, 60);
            return false;
        }
    }

    public boolean turnOffDevice(Device device, boolean manual) {
        DeviceController deviceController = DeviceControllerFactory.createDeviceController((Device)device, (EventEntryManagerService)this.eventEntryManagerService, (ShellyRestService)this.shellyRestService);
        assert (deviceController != null);
        if (deviceController.turnOffDevice()) {
            device.getDeviceStatus().setLastStateChange(LocalDateTime.now());
            this.setTodayRuntime(device);
            device.getDeviceStatus().setCurrentRuntime(0L);
            if (manual) {
                device.getDeviceStatus().setManualOverride(true);
                device.getDeviceStatus().setOverriddenUntil(LocalDateTime.now().plusHours(1L));
            }
            device.getDeviceStatus().setCurrentState(DeviceStatus.OperationStatus.OFF);
            this.saveDeviceStatus(device.getDeviceStatus());
            return true;
        }
        return false;
    }

    @Transactional
    public void dailyReset(Device device, DeviceController deviceController) {
        LocalDate today = LocalDate.now();
        LocalDate lastReset = device.getDeviceStatus().getDailyResetHappened();
        log.info("Daily reset check for device: {} - Today: {}, Last reset: {}", new Object[]{device.getDeviceName(), today, lastReset});
        if (lastReset == null || !lastReset.isEqual(today)) {
            log.info("Performing daily reset for device: {}", (Object)device.getDeviceName());
            float oldTotal = device.getDeviceStatus().getTotalConsumption();
            float todayValue = device.getDeviceStatus().getTodayConsumption();
            float newTotalConsumption = oldTotal + todayValue;
            device.getDeviceStatus().setTotalConsumption(newTotalConsumption);
            device.getDeviceStatus().setLastDayTotalConsumption(deviceController.getTotalConsumption());
            device.getDeviceStatus().setTodayConsumption(0.0f);
            device.getDeviceStatus().setTodayRuntime(0L);
            device.getDeviceStatus().setDailyResetHappened(today);
            log.info("Daily reset values for {}: Old total: {}, Today consumption: {}, New total: {}, Device baseline: {}", new Object[]{device.getDeviceName(), Float.valueOf(oldTotal), Float.valueOf(todayValue), Float.valueOf(newTotalConsumption), Float.valueOf(deviceController.getTotalConsumption())});
            this.eventEntryManagerService.raiseEvent("DEVICE: Daily reset for device: " + device.getDeviceName() + " - Cumulative total: " + newTotalConsumption + " kWh, Device baseline set to: " + deviceController.getTotalConsumption() + " kWh", EventEntry.EventType.INFO);
            this.saveDeviceStatus(device.getDeviceStatus());
        } else {
            log.debug("Daily reset already happened today for device: {}", (Object)device.getDeviceName());
        }
    }

    public void setTodayRuntime(Device device) {
        if (device.getDeviceStatus().getCurrentState() != DeviceStatus.OperationStatus.ON) {
            return;
        }
        long minuteCount = 0L;
        minuteCount = device.getDeviceStatus().getCurrentRuntime() >= (long)LocalTime.now().get(ChronoField.MINUTE_OF_DAY) ? (long)LocalTime.now().get(ChronoField.MINUTE_OF_DAY) : device.getDeviceStatus().getCurrentRuntime() + device.getDeviceStatus().getTodayRuntime();
        device.getDeviceStatus().setTodayRuntime(minuteCount);
        this.saveDeviceStatus(device.getDeviceStatus());
    }

    public ShellyRestService getShellyRestService() {
        return this.shellyRestService;
    }

    public DeviceSettingManagerService getDeviceSettingManagerService() {
        return this.deviceSettingManagerService;
    }

    public EventEntryManagerService getEventEntryManagerService() {
        return this.eventEntryManagerService;
    }

    public DeviceRepository getDeviceRepository() {
        return this.deviceRepository;
    }

    public DeviceStatusRepository getDeviceStatusRepository() {
        return this.deviceStatusRepository;
    }

    public void setDeviceSettingManagerService(DeviceSettingManagerService deviceSettingManagerService) {
        this.deviceSettingManagerService = deviceSettingManagerService;
    }

    public void setEventEntryManagerService(EventEntryManagerService eventEntryManagerService) {
        this.eventEntryManagerService = eventEntryManagerService;
    }

    public void setDeviceRepository(DeviceRepository deviceRepository) {
        this.deviceRepository = deviceRepository;
    }

    public void setDeviceStatusRepository(DeviceStatusRepository deviceStatusRepository) {
        this.deviceStatusRepository = deviceStatusRepository;
    }

    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        if (!(o instanceof DeviceManagerService)) {
            return false;
        }
        DeviceManagerService other = (DeviceManagerService)o;
        if (!other.canEqual((Object)this)) {
            return false;
        }
        ShellyRestService this$shellyRestService = this.getShellyRestService();
        ShellyRestService other$shellyRestService = other.getShellyRestService();
        if (this$shellyRestService == null ? other$shellyRestService != null : !this$shellyRestService.equals(other$shellyRestService)) {
            return false;
        }
        DeviceSettingManagerService this$deviceSettingManagerService = this.getDeviceSettingManagerService();
        DeviceSettingManagerService other$deviceSettingManagerService = other.getDeviceSettingManagerService();
        if (this$deviceSettingManagerService == null ? other$deviceSettingManagerService != null : !this$deviceSettingManagerService.equals(other$deviceSettingManagerService)) {
            return false;
        }
        EventEntryManagerService this$eventEntryManagerService = this.getEventEntryManagerService();
        EventEntryManagerService other$eventEntryManagerService = other.getEventEntryManagerService();
        if (this$eventEntryManagerService == null ? other$eventEntryManagerService != null : !this$eventEntryManagerService.equals(other$eventEntryManagerService)) {
            return false;
        }
        DeviceRepository this$deviceRepository = this.getDeviceRepository();
        DeviceRepository other$deviceRepository = other.getDeviceRepository();
        if (this$deviceRepository == null ? other$deviceRepository != null : !this$deviceRepository.equals(other$deviceRepository)) {
            return false;
        }
        DeviceStatusRepository this$deviceStatusRepository = this.getDeviceStatusRepository();
        DeviceStatusRepository other$deviceStatusRepository = other.getDeviceStatusRepository();
        return !(this$deviceStatusRepository == null ? other$deviceStatusRepository != null : !this$deviceStatusRepository.equals(other$deviceStatusRepository));
    }

    protected boolean canEqual(Object other) {
        return other instanceof DeviceManagerService;
    }

    public int hashCode() {
        int PRIME = 59;
        int result = 1;
        ShellyRestService $shellyRestService = this.getShellyRestService();
        result = result * 59 + ($shellyRestService == null ? 43 : $shellyRestService.hashCode());
        DeviceSettingManagerService $deviceSettingManagerService = this.getDeviceSettingManagerService();
        result = result * 59 + ($deviceSettingManagerService == null ? 43 : $deviceSettingManagerService.hashCode());
        EventEntryManagerService $eventEntryManagerService = this.getEventEntryManagerService();
        result = result * 59 + ($eventEntryManagerService == null ? 43 : $eventEntryManagerService.hashCode());
        DeviceRepository $deviceRepository = this.getDeviceRepository();
        result = result * 59 + ($deviceRepository == null ? 43 : $deviceRepository.hashCode());
        DeviceStatusRepository $deviceStatusRepository = this.getDeviceStatusRepository();
        result = result * 59 + ($deviceStatusRepository == null ? 43 : $deviceStatusRepository.hashCode());
        return result;
    }

    public String toString() {
        return "DeviceManagerService(shellyRestService=" + this.getShellyRestService() + ", deviceSettingManagerService=" + this.getDeviceSettingManagerService() + ", eventEntryManagerService=" + this.getEventEntryManagerService() + ", deviceRepository=" + this.getDeviceRepository() + ", deviceStatusRepository=" + this.getDeviceStatusRepository() + ")";
    }
}

