/*
 * Decompiled with CFR 0.152.
 */
package cz.snyll.sunny.services;

import cz.snyll.sunny.domain.EventEntry;
import cz.snyll.sunny.services.DatabaseBackupService;
import cz.snyll.sunny.services.EventEntryManagerService;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.sql.Connection;
import java.sql.Statement;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import javax.sql.DataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;

@Service
public class DatabaseBackupService {
    private static final Logger log = LoggerFactory.getLogger(DatabaseBackupService.class);
    private static final DateTimeFormatter FORMATTER = DateTimeFormatter.ofPattern("yyyyMMdd");
    private static final String BACKUP_FOLDER = "./data/backups/";
    private final DataSource dataSource;
    private final EventEntryManagerService eventEntryManagerService;

    @Autowired
    public DatabaseBackupService(DataSource dataSource, EventEntryManagerService eventEntryManagerService) {
        this.dataSource = dataSource;
        this.eventEntryManagerService = eventEntryManagerService;
    }

    @Scheduled(cron="0 0 3 * * ?")
    public void backupDatabase() {
        LocalDate today = LocalDate.now();
        String dateStamp = FORMATTER.format(today);
        try {
            Files.createDirectories(Paths.get(BACKUP_FOLDER, new String[0]), new FileAttribute[0]);
        }
        catch (IOException createDirException) {
            this.eventEntryManagerService.raiseEvent("DB BACKUP: Failed to create backup folder: " + createDirException.getMessage(), EventEntry.EventType.ERROR);
            return;
        }
        try (Connection conn = this.dataSource.getConnection();
             Statement stmt = conn.createStatement();){
            String product = conn.getMetaData().getDatabaseProductName();
            if (!"SQLite".equalsIgnoreCase(product)) {
                this.eventEntryManagerService.raiseEvent("DB BACKUP: Only SQLite is supported (detected: " + product + ")", EventEntry.EventType.ERROR);
                return;
            }
            String backupFileName = "database_backup_" + dateStamp + ".mv.db";
            String backupPath = Paths.get(BACKUP_FOLDER, backupFileName).toString().replace("\\", "/");
            String sql = "VACUUM INTO '" + backupPath + "'";
            stmt.execute(sql);
            this.eventEntryManagerService.raiseEvent("DB BACKUP: Backup completed successfully.", EventEntry.EventType.SUCCESS);
        }
        catch (Exception e) {
            this.eventEntryManagerService.raiseEvent("DB BACKUP: Failed to backup database: " + e.getMessage(), EventEntry.EventType.ERROR);
        }
        this.manageOldBackups();
    }

    private void manageOldBackups() {
        LocalDate today = LocalDate.now();
        LocalDate keepDailySince = today.minusDays(3L);
        LocalDate anchorDay = today.minusDays(1L);
        LocalDate anchorWeek = today.minusDays(7L);
        LocalDate anchorMonth = today.minusDays(30L);
        Path backupFolderPath = Paths.get(BACKUP_FOLDER, new String[0]);
        ArrayList<BackupFile> backups = new ArrayList<BackupFile>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(backupFolderPath, "database_backup_*.mv.db");){
            for (Path path : stream) {
                String fileName = path.getFileName().toString();
                if (!fileName.endsWith(".mv.db") || fileName.length() < 31) continue;
                try {
                    String datePart = fileName.substring(16, 24);
                    LocalDate fileDate = LocalDate.parse(datePart, FORMATTER);
                    backups.add(new BackupFile(path, fileName, fileDate));
                }
                catch (Exception parseException) {
                    log.warn("DB BACKUP: Skipping unrecognized backup filename: {}", (Object)fileName);
                }
            }
        }
        catch (IOException e) {
            log.error("Failed to list backup files", (Throwable)e);
            this.eventEntryManagerService.raiseEvent("DB BACKUP: Failed to list backup files: " + e.getMessage(), EventEntry.EventType.ERROR);
            return;
        }
        if (backups.isEmpty()) {
            return;
        }
        backups.sort(Comparator.comparing(b -> b.date).reversed());
        HashSet<Path> keep = new HashSet<Path>();
        backups.stream().filter(b -> !b.date.isBefore(keepDailySince)).forEach(b -> keep.add(b.path));
        this.keepNearestAnchor(backups, anchorDay, keep);
        this.keepNearestAnchor(backups, anchorWeek, keep);
        this.keepNearestAnchor(backups, anchorMonth, keep);
        keep.add(((BackupFile)backups.get((int)0)).path);
        List toDelete = backups.stream().filter(b -> !keep.contains(b.path)).collect(Collectors.toList());
        for (BackupFile bf : toDelete) {
            try {
                Files.deleteIfExists(bf.path);
                this.eventEntryManagerService.raiseEvent("DB BACKUP: Deleted old backup file: " + bf.fileName, EventEntry.EventType.SUCCESS);
            }
            catch (IOException e) {
                this.eventEntryManagerService.raiseEvent("DB BACKUP: Failed deleting old backup '" + bf.fileName + "': " + e.getMessage(), EventEntry.EventType.ERROR);
            }
        }
    }

    private void keepNearestAnchor(List<BackupFile> backups, LocalDate anchor, Set<Path> keep) {
        BackupFile candidateBefore = backups.stream().filter(b -> !b.date.isAfter(anchor)).max(Comparator.comparing(b -> b.date)).orElse(null);
        BackupFile candidateAfter = backups.stream().filter(b -> b.date.isAfter(anchor)).min(Comparator.comparing(b -> b.date)).orElse(null);
        BackupFile chosen = candidateBefore != null ? candidateBefore : candidateAfter;
        if (chosen != null) {
            keep.add(chosen.path);
        }
    }
}

